<?php

declare(strict_types=1);

namespace Erlage\Photogram\Requests\Admin;

use Erlage\Photogram\Constants\SystemConstants;
use Erlage\Photogram\Pattern\ExceptionalRequests;
use Erlage\Photogram\Data\Tables\Sys\RequestTable;
use Erlage\Photogram\Data\Models\Setting\SettingModel;
use Erlage\Photogram\Data\Tables\Setting\SettingTable;
use Erlage\Photogram\Data\Models\Setting\SettingFinder;
use Erlage\Photogram\Data\Dtos\Admin\AdminSettingSaveResultsDTO;

final class AdminSettings extends ExceptionalRequests
{
    public static function load(): void
    {
        self::$flagAdminMode = true;

        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | ensure admin is authenticated
            |--------------------------------------------------------------------------
            */

            self::adminEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | find beans
            |--------------------------------------------------------------------------
            */

            $finder = (new SettingFinder()) -> find();

            while ($finder -> canPop())
            {
                self::addToResponse(SettingTable::getTableName(), $finder -> popModelFromResults() -> getDataMap());
            }
        });
    }

    public static function save(): void
    {
        self::$flagAdminMode = true;

        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | get data
            |--------------------------------------------------------------------------
            */

            $settings = self::$request -> getCollection(
                SettingTable::ID,
                RequestTable::PAYLOAD,
                SettingTable::TABLE_NAME
            );

            /*
            |--------------------------------------------------------------------------
            | ensure admin is authenticated
            |--------------------------------------------------------------------------
            */

            self::adminEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | validate settings data
            |--------------------------------------------------------------------------
            */

            foreach ($settings as $id => $settingData)
            {
                $validatorResult = AdminSettingsValidator::validate($settingData);

                if (SystemConstants::OK != $validatorResult)
                {
                    // best way to send a DTO with some information.
                    // will rework this in future updates.

                    $result = (new AdminSettingSaveResultsDTO())
                        -> setResultType(SystemConstants::NA)
                        -> setMessage($validatorResult)
                        -> setSettingKey($settingData[SettingTable::KEY]);

                    return self::$response -> setContent(
                        AdminSettingSaveResultsDTO::DTO_NAME,
                        array($result -> jsonSerialize())
                    );
                }
            }

            /*
            |--------------------------------------------------------------------------
            | save settings
            |--------------------------------------------------------------------------
            */

            foreach ($settings as $id => $settingData)
            {
                $settingModel = SettingModel::findFromId_throwException($settingData[SettingTable::ID]);

                $settingModel -> update(array(
                    SettingTable::VALUE => $settingData[SettingTable::VALUE],
                ));

                $settingModel -> save();

                self::addToResponse(SettingTable::getTableName(), $settingModel -> getDataMap());
            }
        });
    }
}
